import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Music, Zap, Heart, Sparkles, Award, TrendingUp, Users, Clock, Star, User, Radio } from 'lucide-react'
import { motion } from 'framer-motion'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedStyle, setSelectedStyle] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const danceStyles = [
    { 
      id: 'hiphop', 
      name: 'Хип-хоп', 
      icon: Zap,
      color: '#E63946',
      description: 'Энергичный уличный танец',
      gradient: 'linear-gradient(135deg, #E63946 0%, #F77F00 100%)',
      duration: '12 занятий',
      level: 'Любой уровень'
    },
    { 
      id: 'bachata', 
      name: 'Бачата', 
      icon: Heart,
      color: '#F77F00',
      description: 'Страстный латиноамериканский танец',
      gradient: 'linear-gradient(135deg, #F77F00 0%, #FCBF49 100%)',
      duration: '10 занятий',
      level: 'Начинающий'
    },
    { 
      id: 'zumba', 
      name: 'Зумба', 
      icon: Sparkles,
      color: '#FCBF49',
      description: 'Фитнес-танец для всех',
      gradient: 'linear-gradient(135deg, #FCBF49 0%, #E63946 100%)',
      duration: '8 занятий',
      level: 'Любой уровень'
    },
    { 
      id: 'ballroom', 
      name: 'Бальные', 
      icon: Award,
      color: '#E63946',
      description: 'Классические бальные танцы',
      gradient: 'linear-gradient(135deg, #E63946 0%, #F77F00 100%)',
      duration: '16 занятий',
      level: 'Продвинутый'
    }
  ]

  const instructors = [
    { name: 'Алексей Воронов', experience: '10 лет', rating: 5, specialty: 'Хип-хоп, Брейк-данс', students: '3000+' },
    { name: 'Мария Соколова', experience: '12 лет', rating: 5, specialty: 'Бачата, Сальса', students: '4500+' },
    { name: 'Елена Петрова', experience: '8 лет', rating: 5, specialty: 'Зумба, Фитнес', students: '6000+' },
    { name: 'Дмитрий Кузнецов', experience: '15 лет', rating: 5, specialty: 'Бальные, Латина', students: '2500+' }
  ]

  const benefits = [
    { icon: Music, title: 'Живая музыка', text: 'Занятия под живую музыку и профессиональные треки' },
    { icon: Zap, title: 'Энергия', text: 'Заряд бодрости и позитивных эмоций' },
    { icon: Users, title: 'Сообщество', text: 'Общение с единомышленниками' },
    { icon: Heart, title: 'Здоровье', text: 'Улучшение физической формы и координации' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedStyle && startDate) {
      const savedClasses = localStorage.getItem('danceClasses')
      const classes = savedClasses ? JSON.parse(savedClasses) : []
      
      const styleData = danceStyles.find(s => s.id === selectedStyle)
      const randomInstructor = instructors[Math.floor(Math.random() * instructors.length)]
      
      const newClass = {
        id: Date.now(),
        style: styleData.name,
        instructor: randomInstructor.name,
        date: startDate,
        rating: 0
      }

      classes.push(newClass)
      localStorage.setItem('danceClasses', JSON.stringify(classes))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedStyle('')
        setStartDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="enroll-page">
      <div className="container">
        <motion.section 
          className="enroll-hero"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7 }}
        >
          <motion.div 
            className="enroll-icon-wrapper"
            initial={{ scale: 0, rotate: -360 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.4 }}
          >
            <Music size={110} strokeWidth={2.5} className="enroll-hero-icon" />
          </motion.div>
          <h1 className="enroll-title">Записаться на занятие</h1>
          <p className="enroll-subtitle">
            Выберите стиль танца и дату начала для вашего танцевального пути
          </p>
        </motion.section>

        <motion.section 
          className="enrollment-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.3 }}
        >
          <div className="enrollment-form-wrapper">
            <form className="enrollment-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  <Music size={36} strokeWidth={2.5} className="label-icon" />
                  Выберите стиль танца
                </label>
                <div className="styles-grid">
                  {danceStyles.map((style) => {
                    const IconComponent = style.icon
                    return (
                      <motion.button
                        key={style.id}
                        type="button"
                        className={`style-card ${selectedStyle === style.id ? 'selected' : ''}`}
                        onClick={() => setSelectedStyle(style.id)}
                        whileHover={{ y: -8, rotate: 3, transition: { duration: 0.3 } }}
                        whileTap={{ scale: 0.95 }}
                        style={{ 
                          borderColor: selectedStyle === style.id ? style.color : 'rgba(255, 0, 110, 0.4)',
                          background: selectedStyle === style.id ? (style.color + '25') : 'rgba(10, 10, 15, 0.95)'
                        }}
                      >
                        <div className="style-icon-wrapper" style={{ background: style.color + '30' }}>
                          <IconComponent size={64} strokeWidth={2.5} color={style.color} />
                        </div>
                        <div className="style-content">
                          <h3 className="style-name">{style.name}</h3>
                          <p className="style-description">{style.description}</p>
                          <div className="style-meta">
                            <span className="style-duration">
                              <Clock size={20} strokeWidth={2} />
                              {style.duration}
                            </span>
                            <span className="style-level">
                              <Star size={20} strokeWidth={2} />
                              {style.level}
                            </span>
                          </div>
                        </div>
                        {selectedStyle === style.id && (
                          <motion.div 
                            className="style-check"
                            initial={{ scale: 0, rotate: -180 }}
                            animate={{ scale: 1, rotate: 0 }}
                            style={{ background: style.gradient }}
                          >
                            <CheckCircle size={36} strokeWidth={2.5} color="white" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="startDate" className="form-label">
                  <Calendar size={36} strokeWidth={2.5} className="label-icon" />
                  Дата начала занятий
                </label>
                <input
                  type="date"
                  id="startDate"
                  className="form-input"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedStyle || !startDate}
                whileHover={{ scale: 1.05, y: -4 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на занятие</span>
                <CheckCircle size={36} strokeWidth={2.5} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.8 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.4 }}
                >
                  <CheckCircle size={48} strokeWidth={2.5} className="success-icon" />
                  <span>Вы успешно записались на занятие!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="benefits-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title">Преимущества занятий</h2>
          <div className="benefits-grid">
            {benefits.map((benefit, index) => {
              const IconComponent = benefit.icon
              return (
                <motion.div
                  key={benefit.title}
                  className="benefit-card"
                  initial={{ opacity: 0, y: 30, rotate: -10 }}
                  animate={{ opacity: 1, y: 0, rotate: 0 }}
                  transition={{ delay: index * 0.15, duration: 0.6, type: "spring" }}
                  whileHover={{ y: -8, rotate: 5, transition: { duration: 0.3 } }}
                >
                  <div className="benefit-icon">
                    <IconComponent size={52} strokeWidth={2.5} color="#E63946" />
                  </div>
                  <h3 className="benefit-title">{benefit.title}</h3>
                  <p className="benefit-text">{benefit.text}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="instructors-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.6 }}
        >
          <h2 className="section-title">Наши преподаватели</h2>
          <div className="instructors-grid">
            {instructors.map((instructor, index) => (
              <motion.div
                key={instructor.name}
                className="instructor-card"
                initial={{ opacity: 0, scale: 0.8, rotate: -15 }}
                animate={{ opacity: 1, scale: 1, rotate: 0 }}
                transition={{ delay: index * 0.15, duration: 0.6, type: "spring" }}
                whileHover={{ y: -8, rotate: 5, transition: { duration: 0.3 } }}
              >
                <div className="instructor-avatar">
                  <User size={52} strokeWidth={2.5} />
                </div>
                <h3 className="instructor-name">{instructor.name}</h3>
                <p className="instructor-experience">{instructor.experience} опыта</p>
                <div className="instructor-rating">
                  {Array.from({ length: 5 }, (_, i) => (
                    <Star key={i} size={20} fill="#E63946" stroke="#E63946" strokeWidth={2.5} />
                  ))}
                </div>
                <p className="instructor-specialty">{instructor.specialty}</p>
                <div className="instructor-students">
                  <Users size={20} strokeWidth={2} />
                  <span>{instructor.students} учеников</span>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default Enroll

